<?php

namespace MongoDB;

use BSON\ObjectId;
use MongoDB\Driver\WriteResult;

/**
 * Result class for a single-document insert operation.
 */
class InsertOneResult
{
    private $writeResult;
    private $insertedId;

    /**
     * Constructor.
     *
     * @param WriteResult $writeResult
     * @param ObjectId    $insertedId
     */
    public function __construct(WriteResult $writeResult, ObjectId $insertedId = null)
    {
        $this->writeResult = $writeResult;
        $this->insertedId = $insertedId;
    }

    /**
     * Return the number of documents that were inserted.
     *
     * This value is undefined if the write was not acknowledged.
     *
     * @see InsertOneResult::isAcknowledged()
     * @return integer
     */
    public function getInsertedCount()
    {
        return $this->writeResult->getInsertedCount();
    }

    /**
     * Return the inserted ID that was generated by the driver.
     *
     * If the inserted document already had an ID (e.g. it was generated by the
     * application), this will be null.
     *
     * @return ObjectId|null
     */
    public function getInsertedId()
    {
        return $this->insertedId;
    }

    /**
     * Return whether this insert was acknowledged by the server.
     *
     * If the insert was not acknowledged, other fields from the WriteResult
     * (e.g. insertedCount) will be undefined.
     *
     * @return boolean
     */
    public function isAcknowledged()
    {
        return $this->writeResult->isAcknowledged();
    }
}
