<?php

namespace MongoDB;

use MongoDB\Driver\WriteResult;

/**
 * Result class for a multi-document write operation.
 */
class InsertManyResult
{
    private $writeResult;
    private $insertedIds;

    /**
     * Constructor.
     *
     * @param WriteResult $writeResult
     * @param array       $insertedIds
     */
    public function __construct(WriteResult $writeResult, array $insertedIds = array())
    {
        $this->writeResult = $writeResult;
        $this->insertedIds = $insertedIds;
    }

    /**
     * Return the number of documents that were inserted.
     *
     * This value is undefined if the write was not acknowledged.
     *
     * @see InsertManyResult::isAcknowledged()
     * @return integer
     */
    public function getInsertedCount()
    {
        return $this->writeResult->getInsertedCount();
    }

    /**
     * Return the map of inserted IDs that were generated by the driver.
     *
     * The index of each ID in the map corresponds to the document's position
     * in bulk operation. If an inserted document already had an ID (e.g. it was
     * generated by the application), it will not be present in this map.
     *
     * @return array
     */
    public function getInsertedIds()
    {
        return $this->insertedIds;
    }

    /**
     * Return whether this insert result was acknowledged by the server.
     *
     * If the insert was not acknowledged, other fields from the WriteResult
     * (e.g. insertedCount) will be undefined.
     *
     * @return boolean
     */
    public function isAcknowledged()
    {
        return $this->writeResult->isAcknowledged();
    }
}
